/*******************************************************************************/
/*  
	This script produces Table 1
	Mongey, Pilossoph, and Weinberg: "Which Workers Bear the Burden of Social Distancing?"
	Journal of Economic Inequality 2021	
    --
	Alexander Weinberg
	February 16, 2021
*/
/*******************************************************************************/

//________________________________________________________________
// Load Clean Monthly CPS data
use "../Data/CPS/recession_clean_cps", clear 

//________________________________________________________________
// LWFH* Changes in employment
preserve
	collapse (sum) employed [pw=compwt], by(month year low_wfh_binary)

	//________________________________________________________________
	// Log employment
	gen E_peak 			= employed if (month==1 & year==2008) | (month==2 & year==2020) 
	gen E_trough 		= employed if (month==1 & year==2010) | (inlist(month, 4, 11) & year==2020) 

	sort low_wfh_binary year month
	by low_wfh_binary: replace E_peak = E_peak[_n-1] if mi(E_peak)
	drop if mi(E_trough)

	//________________________________________________________________
	// First difference
	gen recession 		= "."
	replace recession 	= "Great"  if (month==1 & year==2008) | (month==1 & year==2010)
	replace recession 	= "Covid" if (month==2 & year==2020) | (month==4 & year==2020)
	drop if month==11
	
	gen diff 			= (E_trough - E_peak) / E_peak 

	
	//________________________________________________________________
	// Second difference
	sort recession low_wfh_binary 
	by recession: gen diff_in_diff = diff - diff[_n-1]
	list recession diff_in_diff if !mi(diff_in_diff)
	tempfile wfh
	save `wfh'
restore
	
//________________________________________________________________
// HPP* Changes in employment
collapse (sum) employed [pw=compwt], by(month year high_pp_binary)
	

//________________________________________________________________
// Peak and trough employment
gen E_peak 			= employed if (month==1 & year==2008) | (month==2 & year==2020) | (year==2001)
gen E_trough 		= employed if (month==1 & year==2010) | (inlist(month, 4, 11) & year==2020) | year==2002

sort high_pp_binary year month
by high_pp_binary: replace E_peak = E_peak[_n-1] if mi(E_peak)
drop if mi(E_trough)

//________________________________________________________________
// First difference
gen recession 		= "."
replace recession 	= "Great"  if (month==1 & year==2008) | (month==1 & year==2010)
replace recession 	= "Covid" if (month==2 & year==2020) | (month==4 & year==2020)
drop if month==11

gen diff 			= (E_trough - E_peak) / E_peak 
	
//________________________________________________________________
// Second difference
sort recession high_pp_binary 
by recession: gen diff_in_diff = diff - diff[_n-1]
list recession diff_in_diff if !mi(diff_in_diff)

//________________________________________________________________
// Combine with WFH 
append using `wfh'
rename high_pp_binary group

replace group = 2 if low_wfh_binary == 0
replace group = 3 if low_wfh_binary == 1
drop if mi(group)

//________________________________________________________________
// Third difference
gen diff3 			= diff_in_diff - diff_in_diff[_n+2] if recession=="Covid"

// __________________________________________________
// PREPARE FOR LATEX
gen cat = .
replace cat = 0 if group == 0
replace cat = 1 if group == 1
replace cat = 2 if low_wfh_binary == 0
replace cat = 3 if low_wfh_binary == 1

#delimit;
label define catlabel
00 "LPP"
01 "HPP"
02 "HWFH"
03 "LWFH";
#delimit cr
label values cat catlabel

// __________________________________________________
// Label recessions
gen downturn = .
replace downturn = 9 if recession == "Great"
replace downturn = 8 if recession == "Covid"

#delimit;
label define downlbl
8 "Covid"
9 "Great";
#delimit cr
label values downturn downlbl

// __________________________________________________
// Clean up formatting
replace diff 		= 100 * diff
replace diff_in_diff= 100 * diff_in_diff
replace diff3 		= 100 * diff3

replace E_peak 	 	= E_peak / 1E6
replace E_trough 	= E_trough / 1E6

replace E_peak 	  	= round(E_peak, 0.01)
replace E_trough    = round(E_trough, 0.01)
replace diff 	  	= round(diff, 0.01)
replace diff_in_diff= round(diff_in_diff, 0.01)
replace diff3 		= round(diff3, 0.01)

// __________________________________________________
// Transpose for easier viewing
drop group year month employed low_wfh_binary recession
rename cat group
xpose, clear varname 

rename v1 covid_LPP
rename v2 covid_HPP
rename v3 great_LPP
rename v4 great_HPP

rename v5 covid_HWFH
rename v6 covid_LWFH
rename v7 great_HWFH
rename v8 great_LWFH

order _varname great_?WFH covid_?WFH great_?PP covid_?PP
drop if inlist(_varname, "group", "downturn")

export delimited "../Tables/table1.csv", replace
	

//end
